<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;

class MemberController extends Controller
{
    // Menampilkan daftar anggota
    public function index()
    {
        $users = User::all();
        $title = 'Members';
        return view('admin.members.index', compact('title', 'users'));
    }

    // Menampilkan form untuk membuat anggota baru
    public function create()
    {
        $title = 'Tambah User';
        return view('admin.members.create', compact('title'));
    }

    // Menyimpan anggota baru
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required',
            'email' => 'required|email|unique:users',
            'role' => 'required',
            'password' => 'required|min:6',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        $request['password'] = bcrypt($request->password);

        if ($request->hasFile('image')) {
            $request['image'] = $request->file('image')->store('images', 'public');
        }

        User::create($request->all());
        return redirect()->route('members.index')->with('success', 'User berhasil ditambahkan!');
    }

    // Menampilkan form untuk mengedit anggota
    public function edit(User $user)
    {
        $title = 'Edit User ' . $user->name;
        return view('admin.members.edit', compact('title', 'user'));
    }

    // Memperbarui data anggota
    public function update(Request $request, User $member)
    {
        $validatedData = $request->validate([
            'name' => 'required|string|max:255',
            'username' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'role' => 'required|string|max:255',
            'password' => 'nullable|min:6',
        ]);

        if ($request->filled('password')) {
            $validatedData['password'] = bcrypt($request->password);
        } else {
            unset($validatedData['password']);
        }

        $member->update($validatedData);

        return redirect()->route('members.index')->with('success', 'Data user berhasil diperbarui.');
    }

    // Menghapus anggota
    public function destroy(User $user)
    {
        $user->delete();
        return redirect()->route('members.index')->with('success', 'User berhasil dihapus!');
    }
}
