<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use App\Models\Role; // Pastikan Anda memiliki model Role ini dan mengimpornya

class CheckUserRole
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     * @param  string  ...$roles  Daftar nama peran yang diizinkan (e.g., 'admin', 'front_office')
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function handle(Request $request, Closure $next, ...$roles): Response
    {
        // 1. Periksa apakah pengguna sudah login. Jika belum, arahkan ke halaman login.
        if (!Auth::check()) {
            return redirect()->route('login');
        }

        $user = Auth::user();
        $userRoleId = $user->role_id; // Ambil ID peran pengguna yang sedang login

        // 2. Dapatkan ID peran untuk 'admin' dari database secara dinamis.
        // Ini memastikan bahwa jika ID admin berubah di DB, middleware tetap berfungsi.
        $adminRoleId = Role::where('nama_role', 'Admin')->value('id');

        // 3. Jika pengguna adalah Admin, izinkan akses ke mana saja.
        // Admin memiliki hak akses tertinggi, jadi mereka tidak perlu diperiksa lebih lanjut.
        if ($adminRoleId && $userRoleId === $adminRoleId) {
            return $next($request);
        }

        // 4. Kumpulkan ID peran yang diizinkan untuk rute ini berdasarkan nama peran ($roles)
        // yang diberikan di rute, dengan mengambil ID-nya dari database.
        $allowedRoleIds = [];
        if (!empty($roles)) {
            // Ambil semua ID peran dari database yang namanya ada dalam daftar $roles
            $dbRoles = Role::whereIn(DB::raw('LOWER(nama_role)'), array_map('strtolower', $roles))->pluck('id')->toArray();
            $allowedRoleIds = $dbRoles;
        }

        // 5. Periksa apakah ID peran pengguna saat ini ada dalam daftar ID peran yang diizinkan.
        // Jika ya, izinkan permintaan untuk dilanjutkan.
        if (in_array($userRoleId, $allowedRoleIds)) {
            return $next($request);
        }

        // 6. Jika pengguna tidak memiliki peran yang diizinkan dan bukan admin,
        // middleware tidak akan melakukan redirect eksplisit.
        // Ini akan menyebabkan Laravel menampilkan halaman 403 default
        // atau error lain jika tidak ada penanganan lebih lanjut.
        // return redirect()->route('unauthorized'); // Baris ini telah dihapus
        abort(403, 'Unauthorized action.'); // Menggunakan abort(403) adalah praktik yang lebih baik daripada membiarkan request "jatuh"
    }
}
