<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Role; // Make sure Role model is imported
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use App\Mail\VerificationCodeMail;
use Illuminate\Support\Str;
use Carbon\Carbon;

class AuthController extends Controller
{
    public function showLoginForm()
    {
        return view('auth.login');
    }

    public function showRegisterForm()
    {
        return view('auth.register');
    }

    public function login(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required',
        ]);

        $credentials = $request->only('email', 'password');

        if (Auth::attempt($credentials)) {
            $request->session()->regenerate();

            $user = Auth::user();

            // Hanya role_id = 3 (Pengguna) yang harus dicek email_verified_at
            if ($user->role_id === 3 && $user->email_verified_at === null) {
                Auth::logout();
                return redirect()->route('verification.notice', ['email' => $user->email])
                    ->with('info', 'Silakan verifikasi email Anda untuk melanjutkan.');
            }

            // Redirect sesuai role
            if (
                $user->role->nama_role === 'Admin' ||
                $user->role->nama_role === 'Front Office' ||
                $user->role->nama_role === 'Penelaah'
            ) {
                return redirect()->route('dashboard')->with('success', 'Selamat datang, ' . $user->nama . '!');
            } elseif ($user->role->nama_role === 'Pengguna') {
                return redirect()->route('home')->with('success', 'Selamat datang kembali!');
            } else {
                Auth::logout();
                return redirect()->route('login')->with('error', 'Akun Anda tidak memiliki akses yang valid.');
            }
        }

        return back()->withErrors([
            'email' => 'Email atau password salah. Silakan coba lagi.',
        ])->withInput();
    }

    public function register(Request $request)
    {
        $request->validate([
            'nama' => 'required|string|max:255',
            'no_telp' => 'required|string|max:20',
            'email' => 'required|email|unique:users',
            'password' => 'required|confirmed|min:6',
        ]);

        $penggunaRoleId = Role::where('nama_role', 'Pengguna')->value('id');
        $verificationCode = str_pad(random_int(0, 999999), 6, '0', STR_PAD_LEFT);

        $user = User::create([
            'nama' => $request->nama,
            'no_telp' => $request->no_telp,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'role_id' => $penggunaRoleId,
            'email_verified_at' => null,
            'verification_code' => $verificationCode,
        ]);

        // Kirim email verifikasi
        Mail::to($user->email)->send(new VerificationCodeMail($verificationCode, $user->nama));

        // Redirect ke halaman verifikasi email, bawa email sebagai query string
        return redirect()->route('verification.notice', ['email' => $user->email])
            ->with('success', 'Pendaftaran berhasil! Silakan cek email Anda untuk kode verifikasi.');
    }

    public function showVerificationForm(Request $request)
    {
        $email = $request->query('email');
        if (!$email) {
            return redirect()->route('login')->with('error', 'Email tidak ditemukan.');
        }
        return view('auth.verify', ['email' => $email]);
    }

    public function resendVerificationCode(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
        ]);

        $user = User::where('email', $request->email)->first();
        if (!$user) {
            return back()->withErrors(['email' => 'Email tidak ditemukan.']);
        }

        if ($user->email_verified_at !== null) {
            return redirect()->route('login')->with('info', 'Email Anda sudah diverifikasi.');
        }

        $verificationCode = str_pad(random_int(0, 999999), 6, '0', STR_PAD_LEFT);
        $user->verification_code = $verificationCode;
        $user->save();

        Mail::to($user->email)->send(new VerificationCodeMail($verificationCode, $user->nama));

        return back()->with('success', 'Kode verifikasi baru telah dikirim ke email Anda.');
    }

    public function verifyCode(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'verification_code' => 'required|digits:6',
        ]);

        $user = User::where('email', $request->email)->first();
        if (!$user) {
            return back()->withErrors(['email' => 'Email tidak ditemukan.']);
        }

        if ($user->verification_code === $request->verification_code) {
            $user->email_verified_at = now();
            $user->verification_code = null;
            $user->save();

            return redirect()->route('login')->with('success', 'Email Anda berhasil diverifikasi! Silakan login.');
        }

        return back()->withErrors(['verification_code' => 'Kode verifikasi salah. Silakan coba lagi.']);
    }

    public function logout(Request $request)
    {
        Auth::logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect()->route('login')->with('success', 'Anda telah berhasil logout.');
    }
}
